from pyparsing import Literal, OneOrMore, Word, printables

from codeable_detectors.basic_detectors import AtLeastOneFileMatchesDetector, \
    get_var_assignment_matches_containing_url_alias, \
    contains_alias, Detector
from codeable_detectors.evidences import FailedEvidence, LinkEvidence
from codeable_detectors.java.java_detectors import get_java_variable_assignments
from codeable_detectors.js.jsDetectors import JSInvocation, JSCallSyncAsyncDetector
from codeable_detectors.pyparsing_patterns import js_string
from codeable_detectors.utils import get_required_keyword_arg


# detects JS request use, not yet considering if it is part of a Promise or not
class JSRequestUse(AtLeastOneFileMatchesDetector):
    def __init__(self):
        super().__init__()
        self.request_detector = JSInvocation("request")
        self.first_arg_pattern = Literal("request") + Literal("(") + OneOrMore(
            js_string | Word(printables, excludeChars=",);"))
        self.file_endings = ["js"]

    def detect_in_context(self, ctx, **kwargs):
        target = get_required_keyword_arg('target', kwargs)
        matches = []

        request_evidence = self.request_detector.detect_in_context(ctx, **kwargs)
        if not request_evidence.has_succeeded():
            return FailedEvidence("JS request use not not found")

        var_assignments_matching_aliases = get_var_assignment_matches_containing_url_alias(
            get_java_variable_assignments(ctx), target.aliases)

        for request_match in request_evidence.matches:
            parameters = request_match.text[request_match.text.find("(") + 1:request_match.text.find(")")]
            if contains_alias(parameters, target.aliases):
                matches.append(request_match)
            else:
                # maybe a variable in the HTTP post method contains the url to the alias
                for var, var_assignment_match in var_assignments_matching_aliases:
                    if var in parameters:
                        matches.append(request_match)
                        matches.append(var_assignment_match)
        if matches:
            return LinkEvidence(matches).set_properties(detector_link_types="restfulHTTP",
                                                        detector_technology_types=["javascript", "request"],
                                                        kwargs=kwargs)
        return FailedEvidence("JS request use not found")


class JSRequestLink(Detector):
    def __init__(self):
        super().__init__()
        self.sync_async_request_detector = JSCallSyncAsyncDetector(JSRequestUse())

    def detect(self, directory, **kwargs):
        return self.sync_async_request_detector.detect(directory, **kwargs)
